define(['RequestService'], function (RequestService) {
    'use strict';

    describe('FHIR Request Service', function () {
        var requestService,
            httpMock;

        beforeEach(function () {
            httpMock = jasmine.createSpyObj('httpMock', ['$http']);

            requestService = RequestService(httpMock.$http);
        });

        describe('findInArray function', function () {
            it('should throw an error if the first parameter is undefined', function () {
                try {
                    requestService.findInArray();
                    fail('Error should have been thrown');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('Array expected.  Usage: findInArray(array, callbackFn)');
                }
            });

            it('should throw an error if the first parameter is not an array', function () {
                try {
                    requestService.findInArray('Not an array');
                    fail('Error should have been thrown');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('Array expected.  Usage: findInArray(array, callbackFn)');
                }
            });

            it('should throw an error if the second parameter is undefined', function () {
                try {
                    requestService.findInArray([]);
                    fail('Error should have been thrown');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('Function expected.  Usage: findInArray(array, callbackFn)');
                }
            });

            it('should throw an error if the second parameter is not a function', function () {
                try {
                    requestService.findInArray([], 'Not a function');
                    fail('Error should have been thrown');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('Function expected.  Usage: findInArray(array, callbackFn)');
                }
            });

            it('should return undefined if the array has no elements', function () {
                var foundItem = requestService.findInArray([], function () {});

                expect(foundItem).toEqual(undefined);
            });

            it('should return undefined if the item is not found', function () {
                var array = ['apple', 'orange', 'banana'];
                var foundItem = requestService.findInArray(array, function (item) {
                    return item === 'grape';
                });

                expect(foundItem).toEqual(undefined);
            });

            it('should return the item if the item is found', function () {
                var array = ['apple', 'orange', 'banana'];
                var foundItem = requestService.findInArray(array, function (item) {
                    return item === 'orange';
                });

                expect(foundItem).toEqual('orange');
            });
        });

        describe('initParams function', function () {
            it('should create a params object if none exists', function () {
                var httpConfig = {};

                requestService.initParams(httpConfig);

                expect(httpConfig.params).toEqual({_format: 'json'});
            });

            it('should extend the current params object', function () {
                var httpConfig = {
                    params: {
                        someParam: 'someValue'
                    }
                };

                requestService.initParams(httpConfig);

                expect(httpConfig.params).toEqual({someParam: 'someValue', _format: 'json'});
            });

            it('should remove the startDate from params and add a date query parameter to the url if startDate is defined', function () {
                var httpConfig = {
                    url: 'http://localhost/fhir/Observation',
                    params: {
                        startDate: '2017-05-03'
                    }
                };

                requestService.initParams(httpConfig);

                expect(httpConfig.params.startDate).toBeFalsy();
                expect(httpConfig.url).toEqual('http://localhost/fhir/Observation?date=ge2017-05-03');
            });

            it('should remove the endDate from params and add a date query parameter to the url if endDate is defined', function () {
                var httpConfig = {
                    url: 'http://localhost/fhir/Observation',
                    params: {
                        endDate: '2017-05-03'
                    }
                };

                requestService.initParams(httpConfig);

                expect(httpConfig.params.endDate).toBeFalsy();
                expect(httpConfig.url).toEqual('http://localhost/fhir/Observation?date=le2017-05-03');
            });

            it('should add both startDate and endDate to the url gracefully if both are defined', function () {
                var httpConfig = {
                    url: 'http://localhost/fhir/Observation',
                    params: {
                        startDate: '2016-05-03',
                        endDate: '2017-05-03'
                    }
                };

                requestService.initParams(httpConfig);
                
                expect(httpConfig.url).toEqual('http://localhost/fhir/Observation?date=ge2016-05-03&date=le2017-05-03');
            });
        });

        describe('transformRequest function', function () {
            var headers = {
                Authorization: 'Bearer someToken',
                SomeHeader: 'SomeValue'
            };

            var getHeaders = function () {
                return headers;
            };

            it('should transform the headers object', function () {
                requestService.transformRequest(undefined, getHeaders);

                expect(headers).toEqual({
                    'Content-Type': 'application/json',
                    SomeHeader: 'SomeValue'
                });
            });

            it('should return undefined if no data object is present', function () {
                var data = requestService.transformRequest(undefined, getHeaders);

                expect(data).toBeUndefined();
            });

            it('should stringify the data object', function () {
                var dataIn = {
                    someKey: 'someValue',
                    someOtherKey: 'someOtherValue'
                };
                var dataOut = requestService.transformRequest(dataIn, getHeaders);

                expect(dataOut).toEqual(JSON.stringify(dataIn));
            });
        });

        describe('sendRequest function', function () {
            beforeEach(function () {
                spyOn(requestService, 'initParams');
            });

            it('should update the config and send the request', function () {
                var httpConfig = {};

                requestService.sendRequest(httpConfig);

                expect(requestService.initParams).toHaveBeenCalledWith(httpConfig);

                expect(httpConfig.transformRequest).toEqual(requestService.transformRequest);

                expect(httpMock.$http).toHaveBeenCalledWith(httpConfig);
            });
        });
    });
});